<?php

/**
 * StatusPage API handler.
 *
 * Handles all API communication with StatusPage.me.
 *
 * @package StatusPageWidget
 */

// Exit if accessed directly.
if (! defined('ABSPATH')) {
    exit;
}

/**
 * Class StatusPage_API
 *
 * Handles fetching and caching data from the StatusPage.me API.
 */
class StatusPage_API
{

    /**
     * Cache duration in seconds.
     *
     * @var int
     */
    const CACHE_DURATION = 30;

    /**
     * Initialize the API class.
     */
    public static function init()
    {
        // Register REST API endpoint for block editor preview.
        add_action('rest_api_init', array(__CLASS__, 'register_rest_routes'));
    }

    /**
     * Register REST API routes.
     */
    public static function register_rest_routes()
    {
        register_rest_route(
            'statuspage-widget/v1',
            '/status/(?P<slug>[a-z0-9]+(?:-[a-z0-9]+)*)',
            array(
                'methods'             => 'GET',
                'callback'            => array(__CLASS__, 'rest_get_status'),
                'permission_callback' => function () {
                    return current_user_can('edit_posts');
                },
                'args'                => array(
                    'slug' => array(
                        'required'          => true,
                        'sanitize_callback' => array(__CLASS__, 'sanitize_slug'),
                        'validate_callback' => array(__CLASS__, 'validate_slug'),
                    ),
                    'inc'  => array(
                        'default'           => 1,
                        'sanitize_callback' => 'absint',
                    ),
                    'rtm'  => array(
                        'default'           => 60,
                        'sanitize_callback' => 'absint',
                    ),
                    'upd'  => array(
                        'default'           => 7,
                        'sanitize_callback' => 'absint',
                    ),
                ),
            )
        );
    }

    /**
     * Sanitize a slug.
     *
     * @param string $slug The slug to sanitize.
     * @return string Sanitized slug.
     */
    public static function sanitize_slug($slug)
    {
        return strtolower(sanitize_text_field($slug));
    }

    /**
     * Validate a slug format.
     *
     * Only allows lowercase letters (a-z), numbers (0-9), and single hyphens between characters.
     *
     * @param string          $slug    The slug to validate.
     * @param WP_REST_Request $request The REST request.
     * @param string          $param   The parameter name.
     * @return bool|WP_Error True if valid, WP_Error otherwise.
     */
    public static function validate_slug($slug, $request, $param)
    {
        if (empty($slug)) {
            return new WP_Error(
                'invalid_slug',
                __('Status page slug is required.', 'status-page-widget'),
                array('status' => 400)
            );
        }

        // Pattern: lowercase letters/numbers, single hyphens between them only
        if (! preg_match('/^[a-z0-9]+(-[a-z0-9]+)*$/', strtolower($slug))) {
            return new WP_Error(
                'invalid_slug_format',
                __('Invalid slug format. Only lowercase letters, numbers, and single hyphens between characters are allowed.', 'status-page-widget'),
                array('status' => 400)
            );
        }

        return true;
    }

    /**
     * REST API callback to get status.
     *
     * @param WP_REST_Request $request The REST request.
     * @return WP_REST_Response|WP_Error
     */
    public static function rest_get_status($request)
    {
        $slug = $request->get_param('slug');
        $inc  = $request->get_param('inc');
        $rtm  = $request->get_param('rtm');
        $upd  = $request->get_param('upd');

        $data = self::get_status($slug, $inc, $rtm, $upd);

        if (is_wp_error($data)) {
            return $data;
        }

        return rest_ensure_response($data);
    }

    /**
     * Get status data from the API.
     *
     * @param string $slug Status page slug.
     * @param int    $inc  Include incidents (1 or 0).
     * @param int    $rtm  Response time minutes (0-240).
     * @param int    $upd  Uptime days (0-30).
     * @return array|WP_Error Status data or error.
     */
    public static function get_status($slug, $inc = 1, $rtm = 60, $upd = 7)
    {
        if (empty($slug)) {
            return new WP_Error('missing_slug', __('Status page slug is required.', 'status-page-widget'));
        }

        // Generate cache key.
        $cache_key = sprintf('statuspage_status_%s_%d_%d_%d', $slug, $inc, $rtm, $upd);

        // Check cache first.
        $cached = get_transient($cache_key);
        if (false !== $cached) {
            return $cached;
        }

        // Build API URL.
        $api_url = add_query_arg(
            array(
                'slug' => sanitize_text_field($slug),
                'inc'  => absint($inc),
                'rtm'  => min(absint($rtm), 240),
                'upd'  => min(absint($upd), 30),
            ),
            STATUSPAGE_API_BASE . '/api/embed/status'
        );

        // Fetch data from API.
        $response = wp_remote_get(
            $api_url,
            array(
                'timeout'    => 10,
                'user-agent' => 'StatusPage-Widget/' . STATUSPAGE_WIDGET_VERSION . ' WordPress/' . get_bloginfo('version'),
            )
        );

        // Handle errors.
        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);

        if (200 !== $status_code) {
            return new WP_Error(
                'api_error',
                sprintf(
                    /* translators: %d: HTTP status code */
                    __('StatusPage API returned error code: %d', 'status-page-widget'),
                    $status_code
                )
            );
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('json_error', __('Failed to parse API response.', 'status-page-widget'));
        }

        // Cache the response.
        set_transient($cache_key, $data, self::CACHE_DURATION);

        return $data;
    }

    /**
     * Get the SVG badge URL for a status page.
     *
     * @param string $slug Status page slug.
     * @return string Badge URL.
     */
    public static function get_badge_url($slug)
    {
        return STATUSPAGE_API_BASE . '/badge/' . sanitize_text_field($slug) . '.svg';
    }

    /**
     * Get the status page public URL.
     *
     * @param string $slug Status page slug.
     * @return string Status page URL.
     */
    public static function get_status_page_url($slug)
    {
        $slug = sanitize_text_field($slug);
        return 'https://' . $slug . '.statuspage.me';
    }

    /**
     * Clear cached data for a specific slug.
     *
     * @param string $slug Status page slug.
     */
    public static function clear_cache($slug)
    {
        global $wpdb;
        $pattern = '_transient_statuspage_status_' . sanitize_text_field($slug) . '_%';
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                $pattern
            )
        );
    }
}
